<?php
namespace Elementor;

use Elementor\Core\Kits\Manager;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor maintenance.
 *
 * Elementor maintenance handler class is responsible for setting up Elementor
 * activation and uninstallation hooks.
 *
 * @since 1.0.0
 */
class Maintenance {

	/**
	 * Activate Elementor.
	 *
	 * Set Elementor activation hook.
	 *
	 * Fired by `register_activation_hook` when the plugin is activated.
	 *
	 * @since 1.0.0
	 * @access public
	 * @static
	 */
	public static function activation( $network_wide ) {
		wp_clear_scheduled_hook( 'elementor/tracker/send_event' );

		wp_schedule_event( time(), 'daily', 'elementor/tracker/send_event' );
		flush_rewrite_rules();

		if ( is_multisite() && $network_wide ) {
			static::create_default_kit(
				get_sites( [ 'fields' => 'ids' ] )
			);

			return;
		}

		static::create_default_kit();

		set_transient( 'elementor_activation_redirect', true, MINUTE_IN_SECONDS );
	}

	/**
	 * Uninstall Elementor.
	 *
	 * Set Elementor uninstallation hook.
	 *
	 * Fired by `register_uninstall_hook` when the plugin is uninstalled.
	 *
	 * @since 1.0.0
	 * @access public
	 * @static
	 */
	public static function uninstall() {
		wp_clear_scheduled_hook( 'elementor/tracker/send_event' );
	}

	/**
	 * Init.
	 *
	 * Initialize Elementor Maintenance.
	 *
	 * @since 1.0.0
	 * @access public
	 * @static
	 */
	public static function init() {
		register_activation_hook( ELEMENTOR_PLUGIN_BASE, [ __CLASS__, 'activation' ] );
		register_uninstall_hook( ELEMENTOR_PLUGIN_BASE, [ __CLASS__, 'uninstall' ] );

		add_action( 'wpmu_new_blog', function ( $site_id ) {
			if ( ! is_plugin_active_for_network( ELEMENTOR_PLUGIN_BASE ) ) {
				return;
			}

			static::create_default_kit( [ $site_id ] );
		} );
	}

	/**
	 * @param array $site_ids
	 */
	private static function create_default_kit( array $site_ids = [] ) {
		if ( ! empty( $site_ids ) ) {
			foreach ( $site_ids as $site_id ) {
				switch_to_blog( $site_id );

				Manager::create_default_kit();

				restore_current_blog();
			};

			return;
		}

		Manager::create_default_kit();
	}
}
