<?php

namespace Elementor;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Elementor heading widget.
 *
 * Elementor widget that displays an eye-catching headlines.
 *
 * @since 1.0.0
 */
class  OSF_Elementor_Heading extends Widget_Heading {

    public function get_title() {
        return __('Opal Heading', 'worldlife-core');
    }

    protected function _register_controls() {
        $this->start_controls_section(
            'section_title',
            [
                'label' => __('Title', 'worldlife-core'),
            ]
        );

        $this->add_control(
            'title',
            [
                'label'       => __('Title', 'worldlife-core'),
                'type'        => Controls_Manager::TEXTAREA,
                'dynamic'     => [
                    'active' => true,
                ],
                'placeholder' => __('Enter your title', 'worldlife-core'),
                'default'     => __('Add Your Heading Text Here', 'worldlife-core'),
            ]
        );

        $this->add_control(
            'link',
            [
                'label'     => __('Link', 'worldlife-core'),
                'type'      => Controls_Manager::URL,
                'dynamic'   => [
                    'active' => true,
                ],
                'default'   => [
                    'url' => '',
                ],
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'size',
            [
                'label'   => __('Size', 'worldlife-core'),
                'type'    => Controls_Manager::SELECT,
                'default' => 'default',
                'options' => [
                    'default' => __('Default', 'worldlife-core'),
                    'small'   => __('Small', 'worldlife-core'),
                    'medium'  => __('Medium', 'worldlife-core'),
                    'large'   => __('Large', 'worldlife-core'),
                    'xl'      => __('XL', 'worldlife-core'),
                    'xxl'     => __('XXL', 'worldlife-core'),
                ],
            ]
        );

        $this->add_control(
            'header_size',
            [
                'label'   => __('HTML Tag', 'worldlife-core'),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    'h1'   => 'H1',
                    'h2'   => 'H2',
                    'h3'   => 'H3',
                    'h4'   => 'H4',
                    'h5'   => 'H5',
                    'h6'   => 'H6',
                    'div'  => 'div',
                    'span' => 'span',
                    'p'    => 'p',
                ],
                'default' => 'h2',
            ]
        );

        $this->add_responsive_control(
            'align',
            [
                'label'     => __('Alignment', 'worldlife-core'),
                'type'      => Controls_Manager::CHOOSE,
                'options'   => [
                    'left'    => [
                        'title' => __('Left', 'worldlife-core'),
                        'icon'  => 'fa fa-align-left',
                    ],
                    'center'  => [
                        'title' => __('Center', 'worldlife-core'),
                        'icon'  => 'fa fa-align-center',
                    ],
                    'right'   => [
                        'title' => __('Right', 'worldlife-core'),
                        'icon'  => 'fa fa-align-right',
                    ],
                    'justify' => [
                        'title' => __('Justified', 'worldlife-core'),
                        'icon'  => 'fa fa-align-justify',
                    ],
                ],
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}}' => 'text-align: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'show_decor',
            [
                'label' => __('Heading Decor', 'worldlife-core'),
                'type'  => Controls_Manager::SWITCHER,
            ]
        );

        $this->add_control(
            'view',
            [
                'label'   => __('View', 'worldlife-core'),
                'type'    => Controls_Manager::HIDDEN,
                'default' => 'traditional',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_title_style',
            [
                'label' => __('Title', 'worldlife-core'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label'     => __('Text Color', 'worldlife-core'),
                'type'      => Controls_Manager::COLOR,
                'scheme'    => [
                    'type'  => Scheme_Color::get_type(),
                    'value' => Scheme_Color::COLOR_4,
                ],
                'selectors' => [
                    // Stronger selector to avoid section style from overwriting
                    '{{WRAPPER}}.elementor-widget-heading .elementor-heading-title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'typography',
                'scheme'   => Scheme_Typography::TYPOGRAPHY_1,
                'selector' => '{{WRAPPER}} .elementor-heading-title',
            ]
        );

        $this->add_group_control(
            Group_Control_Text_Shadow::get_type(),
            [
                'name'     => 'text_shadow',
                'selector' => '{{WRAPPER}} .elementor-heading-title',
            ]
        );

        $this->add_control(
            'blend_mode',
            [
                'label'     => __('Blend Mode', 'worldlife-core'),
                'type'      => Controls_Manager::SELECT,
                'options'   => [
                    ''            => __('Normal', 'worldlife-core'),
                    'multiply'    => 'Multiply',
                    'screen'      => 'Screen',
                    'overlay'     => 'Overlay',
                    'darken'      => 'Darken',
                    'lighten'     => 'Lighten',
                    'color-dodge' => 'Color Dodge',
                    'saturation'  => 'Saturation',
                    'color'       => 'Color',
                    'difference'  => 'Difference',
                    'exclusion'   => 'Exclusion',
                    'hue'         => 'Hue',
                    'luminosity'  => 'Luminosity',
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-heading-title' => 'mix-blend-mode: {{VALUE}}',
                ],
                'separator' => 'none',
            ]
        );
        $this->add_control(
            'decor_color',
            [
                'label'     => __('Decor Color', 'worldlife-core'),
                'type'      => Controls_Manager::COLOR,
                'scheme'    => [
                    'type'  => Scheme_Color::get_type(),
                    'value' => Scheme_Color::COLOR_1,
                ],
                'selectors' => [
                    '{{WRAPPER}} .heading-decor svg' => 'fill: {{VALUE}};',
                ],
                'condition' => [
                    'show_decor' => 'yes'
                ]
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render heading widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since  1.0.0
     * @access protected
     */
    protected function render() {
        $settings = $this->get_settings_for_display();

        if (empty($settings['title'])) {
            return;
        }

        $this->add_render_attribute('title', 'class', 'elementor-heading-title');

        if (!empty($settings['size'])) {
            $this->add_render_attribute('title', 'class', 'elementor-size-' . $settings['size']);
        }

        $this->add_inline_editing_attributes('title');

        $title = $settings['title'];

        if (!empty($settings['link']['url'])) {
            $this->add_render_attribute('url', 'href', $settings['link']['url']);

            if ($settings['link']['is_external']) {
                $this->add_render_attribute('url', 'target', '_blank');
            }

            if (!empty($settings['link']['nofollow'])) {
                $this->add_render_attribute('url', 'rel', 'nofollow');
            }

            $title = sprintf('<a %1$s>%2$s</a>', $this->get_render_attribute_string('url'), $title);
        }

        $title_html = sprintf('<%1$s %2$s>%3$s</%1$s>', $settings['header_size'], $this->get_render_attribute_string('title'), $title);

        if ($settings['show_decor'] == 'yes') {
            $title_html .= '<div class="heading-decor"><svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="88" height="11" x="0px" y="0px" viewBox="0 0 88 10.7" enable-background="new 0 0 88 10.7" xml:space="preserve"> <path fill-rule="evenodd" clip-rule="evenodd" d="M0.7,6.6C0.4,6.6,0.4,6.4,0,6.4C0,6.3,0.9,6.4,0.7,6.6z M0.4,5.9 	c0.3-0.3,0.6,0,1.1,0C1.3,6,0.9,6,0.4,5.9z M1.7,6.5c-0.3,0.1-0.6,0.1-0.9,0C1,6.3,1.4,6.5,1.7,6.5z M4.8,4.2c-0.5,0.1-1-0.1-1.6,0 	C3.5,3.9,4.4,4.2,4.8,4.2z M16.6,7.7C12.4,7.8,8.1,7.3,4.3,7c0-0.1,0.1-0.1,0.1-0.2C8.2,7.2,13,7.1,16.6,7.7z M5.5,5.5 	C5.4,5.4,5.3,5.3,5.1,5.2C5.8,5.1,6.2,5,6.8,5.1c0,0.4-0.6,0.2-0.7,0.2C7.1,5.9,8.5,5.6,10,5.8c-1.6,0.5-3.7-0.1-5.5,0 	c0-0.1,0-0.2,0-0.2C4.9,5.6,5.3,5.6,5.5,5.5z M7.9,5.2c0.3-0.1,0.8-0.2,1,0C8.7,5.4,8.1,5.5,7.9,5.2z M9.5,6.5 	c-0.2,0.2-1.1,0.3-1.3,0C8.5,6.1,9,6.4,9.5,6.5z M9.1,5.3c0-0.1,0-0.2,0-0.2c0.8,0.4,1.3-0.1,1.9,0.2C10.7,5.7,9.6,5.5,9.1,5.3z 	 M14.3,4.6c-0.6,0.2-1.4,0.2-2.4,0.1c0-0.1,0-0.2,0-0.2C12.7,4.5,13.6,4.4,14.3,4.6z M11.6,6.8c0.3-0.4,1.3-0.1,1.8,0 	C12.9,6.9,12.3,6.9,11.6,6.8z M11.9,5.6c0.3-0.4,0.7-0.3,1.4-0.1C12.9,5.8,12.6,5.6,11.9,5.6z M14.5,5.9c-1,0-1.7,0.4-2.6,0.2 	c0.2-0.3,0.8-0.2,1.4-0.2C14.6,5,17.1,5.7,19,5.5c-0.3,0.3-1.2,0.7-1.8,0.7C16.3,6.3,15.4,5.9,14.5,5.9z M14.5,0.3 	c0.1-0.3,0.5-0.1,0.8-0.1c-0.1,0.2,0.2,0.1,0.2,0.3c0,0.3-0.3,0.1-0.4,0.3C14.9,0.6,15,0.2,14.5,0.3z M16.6,9.4 	c-0.4,1-1.7-0.3-2.7-0.1C14.5,9.1,15.8,9.1,16.6,9.4z M19.2,7c-1.4,0.3-3.3,0.1-4.9-0.1c0.1-0.1,0.1-0.1,0.1-0.2 	C15.9,6.6,17.6,6.4,19.2,7z M15.1,4.7c0.3-0.1,0.6-0.2,1-0.1C16,4.9,15.1,5,15.1,4.7z M15.8,0.7c0.6-0.4,1.4-0.5,2.1-0.7 	C17.9,0.5,16.7,1,15.8,0.7z M17.2,4.8c0-0.1,0-0.2,0-0.2c0.2,0,0.3,0.1,0.4,0.1C17.5,4.9,17.4,4.9,17.2,4.8z M17.9,1.8 	c0.3,0,0.2-0.1,0.4,0.1C18.3,2.2,17.7,2.1,17.9,1.8z M18.4,0.2c0.2,0,0.2-0.1,0.4,0c0,0.1-0.1,0.1-0.1,0.1c0.1,0.1,0.3,0,0.2,0.4 	c0.1-0.1,0.4,0,0.6,0c-0.1,0.4-0.7,0.1-1.1,0.1C18.5,0.4,18.4,0.3,18.4,0.2z M21,4.8C20.2,5,18.7,5.1,18,4.7 	C18.9,4.4,20.2,4.5,21,4.8z M20.1,0.8c0-0.3-0.4-0.5-0.1-0.7c0.4,0.1,0.7,0,1,0.1c0.1,0.2-0.4,0.4-0.3,0.7 	C20.6,0.9,20.4,0.9,20.1,0.8z M22.1,9.5c0.1,0.3-0.7,0.5-1.1,0.2c-0.7,0.3-1.7,0.2-2.8,0.1C19,8.9,20.9,9.9,22.1,9.5z M20.5,2 	c0.3-0.1,0.5-0.2,0.9-0.2C21.6,2.1,20.7,2.2,20.5,2z M21.2,0.9c0.1-0.3,0.6-0.3,0.7-0.7c1,0.1,2.2,0,2.9,0.1 	c-0.2,0.3-1.1,0.2-0.9,0.6C23.2,0.6,22.3,0.9,21.2,0.9z M20.5,7.1c0.1-0.2,0.3-0.2,0.4-0.4c0.5,0.1,1,0.1,1.2,0.3 	C21.7,7.4,20.9,7.2,20.5,7.1z M21.6,4.8c0.4-0.1,1-0.4,1.4-0.2C22.7,4.9,22,5,21.6,4.8z M24.3,2.1c-0.5,0.1-1.3,0.1-1.9,0.1 	C22.3,1.8,23.9,1.7,24.3,2.1z M23.2,7.1c0.3-0.2,1-0.4,1.6-0.1C24.5,7.2,23.6,7.5,23.2,7.1z M24.4,9.9c-0.4,0.3-1.1,0-1.4-0.3 	C23.5,9.5,24.1,9.8,24.4,9.9z M25.3,0.7c0-0.1,0-0.1,0-0.2c0.1,0,0.1,0,0.1-0.1c0.1,0,0.1,0,0.2,0c0,0.1,0,0.1,0.1,0.1 	c0,0.1,0,0.1,0,0.2c-0.1,0-0.1,0-0.1,0.1C25.6,0.8,25.5,0.8,25.3,0.7C25.5,0.8,25.4,0.7,25.3,0.7z M25.8,0.5 	c-0.2-0.2,0.3-0.4,0.6-0.3C26.4,0.5,25.9,0.4,25.8,0.5z M24.9,7c0.9-0.5,2.9-0.2,4.2,0C28.7,7.5,27.4,7,27,7.4 	C26.3,7,25.4,7.5,24.9,7z M26.5,9.9c-0.6,0.4-1.6-0.1-2,0C25,9.5,26.1,9.9,26.5,9.9z M27.3,0.4c0-0.3,0.2-0.3,0.5-0.3 	C27.8,0.4,27.6,0.5,27.3,0.4z M28.4,2.8c0.5-0.5,1.6-0.1,2,0C30,3.3,28.8,2.7,28.4,2.8z M28.5,9.9c-0.2,0.2-0.7,0.3-0.8,0.1 	C27.8,9.8,28.3,9.8,28.5,9.9z M36.6,1.2c0.2-0.2,0.6-0.5,0.9-0.3C37.2,1,37,1.2,36.6,1.2z M37.8,0.5c0.1-0.1,0.1-0.1,0-0.2 	c0.1-0.1,0.4-0.1,0.6-0.1C38.4,0.5,38.1,0.5,37.8,0.5z M39.6,0.4c0.5-0.2,1.9-0.2,2.7,0c1.1-0.3,2.8-0.1,4.3,0 	c-0.9,0.7-2.6,1-4.2,0.6c0.1-0.3,0.7-0.1,0.9-0.3C42,0.1,40.7,0.8,39.6,0.4z M41.4,0.9c-0.1,0.2-0.4,0.2-0.7,0.1 	C40.7,0.6,41.3,0.7,41.4,0.9z M52,10.5c-0.1,0.1-0.1,0.1,0,0.2c-0.2,0.1-0.5,0.1-0.8,0C51.1,10.6,51.7,10.5,52,10.5z M64.6,10.1 	c0,0.4-0.3,0.1-0.6,0.1c-1.9,0.4-4.7,0.6-6.7,0.2c-0.8,0.2-1.7,0.4-2.8,0.3c0-0.1,0.1-0.1,0.1-0.2c0.8-0.1,1.9,0,2.5-0.3 	c0.3,0,0.5,0.1,0.8,0.1c1.8,0.1,4.3-0.3,5.9-0.3C64,9.9,64.9,9.9,64.6,10.1z M84.4,4.5c0-0.3-0.2-0.6,0-0.7c0.1,0.1,0.1,0.1,0.3,0.1 	C84.6,4.2,85,4.6,84.4,4.5z M7.2,6.4C7,6.5,6.8,6.5,6.6,6.4c0.1-0.2,0.4,0,0.4-0.2C7.2,6.3,7,6.3,7.2,6.4z M9.3,4.5 	c0.2-0.3,1,0.1,1.4,0C10.5,4.6,9.7,4.7,9.3,4.5z M12.8,9.3c-0.2,0.2-0.6,0-0.9-0.1c0.2-0.1,0.6-0.2,0.7,0c0.1,0,0.1,0,0.1-0.1 	C12.9,9.1,12.6,9.2,12.8,9.3z M19.5,6.1c0-0.2-0.2-0.2-0.1-0.4c0.2-0.1,0.7,0,0.9,0c-0.2,0.3,0.5,0.2,0.4,0.5 	C20.3,5.6,20.2,6.3,19.5,6.1z M25.7,4.7c0.4-0.5,1.2-0.1,1.7-0.1c0.2,0.1-0.6,0-0.7,0.2c-0.3,0-0.2-0.3-0.4-0.3 	C26.1,4.6,26.1,4.9,25.7,4.7z M87.2,7.6c-0.6,0.7-2.4,0-2.9,0.7c-5.2-0.2-9.2,0.1-14.5-0.2c-0.3,0.1,0.3,0.2,0.1,0.5 	c-0.7,0.2-1.9-0.1-2.5,0.1c0.6,0.5,2.5,0,3,0.4c-0.1,0.4-0.8,0.2-1.3,0.2c-2,0.5-4.3,0.1-6.6,0.1c-1.5,0-3.2,0.1-4.8,0.1 	c-6.2,0.1-12.3,0.3-19,0.6C38.4,10.3,38.5,10,38,10c-1.4-0.1-2.9,0.5-4.4,0.4c-1.2-0.1-2.8-0.5-3.9-0.3c0.2-0.4,1.4-0.2,1.8-0.1 	c3.5-0.6,7.5-0.3,10.8-0.3c1-0.4,2.8,0.1,3.7-0.5c0-0.1-0.1-0.1-0.1-0.2c-0.4-0.3-0.9,0.2-1.4,0.1c-0.3,0-0.1-0.3-0.3-0.4 	c-1,0.1-1.9-0.2-3.1,0c0,0.4,0.5,0,0.7,0.3c-0.9,0.1-2.6,0.6-2.7-0.4c0.1,0.1,0.5,0.2,0.2,0.4c0.5-0.1,0.9-0.1,1.4-0.2 	c-0.1-0.4-0.9,0-1.3,0c-0.1-0.2-0.4-0.2-0.5-0.4C35,8.2,30.5,8.7,26.2,7.9c2.2-0.3,5.2,0.1,7.6-0.4c0.5,0.7,1.7,0.2,2.5,0.3 	c-0.2-0.5-1.3-0.4-1.7-0.2c0.1-0.3-0.6-0.1-0.4-0.4c-1.6-0.1-3.3,0.7-4.7-0.3c0.5-0.3,1.3-0.2,2-0.3c-1.8-0.7-3.7-0.1-5.5-0.2 	c-0.4,0-0.9-0.3-1.4-0.3c-0.4,0-0.8,0.2-1.2,0.2c-0.8,0-1.6-0.5-2.5-0.4c0.1-0.4,0.7-0.2,1.2-0.2c1.4-0.4,2.9-0.1,4.3-0.3 	C27.8,5.4,29,4.6,30.3,5c0.2-0.2,0.5-0.4,0.9-0.4c0,0.2,0.2,0.3,0.6,0.3c0.1-0.2-0.2-0.2-0.3-0.3c0.3-0.3,0.6,0,0.9,0.1 	c0.2,0-0.1-0.3,0.3-0.3c0.4,0,0.3,0.2,0.4,0.3c1.5-0.3,3.3,0.3,4.5-0.3c0.4,0,0.6,0.1,0.9,0.2c0.2,0,0.2-0.3,0.4-0.4 	c3.3,0.3,6.7,0,10.1,0c-0.4-0.2-1.5,0-2.2-0.4c-0.5,0-2.2,0.3-2.5,0.2c-0.1,0-0.2-0.3-0.4-0.3c-0.9-0.1-1.5,0.5-2.7,0.3 	c-0.3,0-0.1-0.3-0.3-0.4c-0.9,0-2,0.1-3.4,0c-0.7,0-1.9-0.3-2.1,0c0,0,0.2,0.2,0.3,0.1c-0.3,0.1-0.8,0.1-1,0.2 	c0-0.3,0.3-0.3,0.1-0.5c-0.9,0.1-1.1-0.3-1.7-0.4c0-0.7,1.2,0,1.4-0.5c-0.8-0.3-2.6-0.6-3.5,0c-1.7-0.9-4,0-5.7-0.4 	c0.4-0.5,1.6-0.5,2.3-0.4c1.3-0.5,3,0.2,4.2-0.3c0.4,0.5,1.1,0.1,1.7,0.4c1.6-0.9,4,0.2,5.8-0.5c0.2,0.1,0.2,0.2,0.5,0.2 	c0.8-0.2,2,0,2.7-0.4c0.8,0.2,1.4-0.1,2,0.2c0.5-0.1,1-0.2,1.7-0.1c0.6-0.4,1.8-0.2,2.3-0.7c-0.4-0.3-0.9,0.2-1.5,0 	c-0.1-0.1,0-0.3-0.2-0.4c2.5-0.4,5.4,0,8.2-0.1c0.2,0.1,0.5,0.2,0.3,0.3c0.4,0,0.5-0.3,0.8-0.3c2.6-0.2,5.3,0.1,7.8,0.3 	c1.8,0.1,3.6-0.2,5.1,0.6c1.4,0.1,3.2,0,4.9,0.5c0.8,0.2,1.3,0.6,2.1,0.6c0.5,0,1.1-0.2,1.6-0.1c0.2,0.1,0.1,0.4,0.2,0.5 	c0.8,0.1,1.5-0.3,2.1,0c0,0.2-0.3,0.1-0.4,0.4C79.7,3,80,3.1,80.2,2.9C80.6,4.3,83.3,4,84,5.2c1.5,0.1,3.9,0.5,4,1.5 	c-0.6-0.2-1.5-0.8-2.2-0.4C85.9,6.9,87.3,7.1,87.2,7.6z M35.7,0.1c0.2,0-0.1,0.1,0.1,0.2c-0.2,0.1-0.6,0.2-0.8,0.2 	C34.2,0.5,34.6,0,35.7,0.1z M53.2,10.6c0.3-0.2,0.7-0.2,1.1-0.1c-0.1,0.4-0.9,0.1-1.3,0.2C53.2,10.6,53.4,10.6,53.2,10.6z M27.3,0.8 	c1.6-0.9,5.1-0.7,7.1-0.6c-1.3,0.6-3.4,1.1-5.2,0.6c0.1-0.2,0.5,0,0.7-0.2C28.9,0.2,28.1,1.3,27.3,0.8z"/></svg></div>';
        }
        echo $title_html;
    }

    /**
     * Render heading widget output in the editor.
     *
     * Written as a Backbone JavaScript template and used to generate the live preview.
     *
     * @since  1.0.0
     * @access protected
     */
    protected function _content_template() {
        ?>
        <#
        var title = settings.title;

        if ( '' !== settings.link.url ) {
        title = '<a href="' + settings.link.url + '">' + title + '</a>';
        }

        view.addRenderAttribute( 'title', 'class', [ 'elementor-heading-title', 'elementor-size-' + settings.size ] );

        view.addInlineEditingAttributes( 'title' );

        var title_html = '<' + settings.header_size  + ' ' + view.getRenderAttributeString( 'title' ) + '>' + title + '</' + settings.header_size + '>';

        if(settings.show_decor !== '') {
        title_html += '<div class="heading-decor"><svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="88" height="11" x="0px" y="0px" viewBox="0 0 88 10.7" enable-background="new 0 0 88 10.7" xml:space="preserve"> <path fill-rule="evenodd" clip-rule="evenodd" d="M0.7,6.6C0.4,6.6,0.4,6.4,0,6.4C0,6.3,0.9,6.4,0.7,6.6z M0.4,5.9 	c0.3-0.3,0.6,0,1.1,0C1.3,6,0.9,6,0.4,5.9z M1.7,6.5c-0.3,0.1-0.6,0.1-0.9,0C1,6.3,1.4,6.5,1.7,6.5z M4.8,4.2c-0.5,0.1-1-0.1-1.6,0 	C3.5,3.9,4.4,4.2,4.8,4.2z M16.6,7.7C12.4,7.8,8.1,7.3,4.3,7c0-0.1,0.1-0.1,0.1-0.2C8.2,7.2,13,7.1,16.6,7.7z M5.5,5.5 	C5.4,5.4,5.3,5.3,5.1,5.2C5.8,5.1,6.2,5,6.8,5.1c0,0.4-0.6,0.2-0.7,0.2C7.1,5.9,8.5,5.6,10,5.8c-1.6,0.5-3.7-0.1-5.5,0 	c0-0.1,0-0.2,0-0.2C4.9,5.6,5.3,5.6,5.5,5.5z M7.9,5.2c0.3-0.1,0.8-0.2,1,0C8.7,5.4,8.1,5.5,7.9,5.2z M9.5,6.5 	c-0.2,0.2-1.1,0.3-1.3,0C8.5,6.1,9,6.4,9.5,6.5z M9.1,5.3c0-0.1,0-0.2,0-0.2c0.8,0.4,1.3-0.1,1.9,0.2C10.7,5.7,9.6,5.5,9.1,5.3z 	 M14.3,4.6c-0.6,0.2-1.4,0.2-2.4,0.1c0-0.1,0-0.2,0-0.2C12.7,4.5,13.6,4.4,14.3,4.6z M11.6,6.8c0.3-0.4,1.3-0.1,1.8,0 	C12.9,6.9,12.3,6.9,11.6,6.8z M11.9,5.6c0.3-0.4,0.7-0.3,1.4-0.1C12.9,5.8,12.6,5.6,11.9,5.6z M14.5,5.9c-1,0-1.7,0.4-2.6,0.2 	c0.2-0.3,0.8-0.2,1.4-0.2C14.6,5,17.1,5.7,19,5.5c-0.3,0.3-1.2,0.7-1.8,0.7C16.3,6.3,15.4,5.9,14.5,5.9z M14.5,0.3 	c0.1-0.3,0.5-0.1,0.8-0.1c-0.1,0.2,0.2,0.1,0.2,0.3c0,0.3-0.3,0.1-0.4,0.3C14.9,0.6,15,0.2,14.5,0.3z M16.6,9.4 	c-0.4,1-1.7-0.3-2.7-0.1C14.5,9.1,15.8,9.1,16.6,9.4z M19.2,7c-1.4,0.3-3.3,0.1-4.9-0.1c0.1-0.1,0.1-0.1,0.1-0.2 	C15.9,6.6,17.6,6.4,19.2,7z M15.1,4.7c0.3-0.1,0.6-0.2,1-0.1C16,4.9,15.1,5,15.1,4.7z M15.8,0.7c0.6-0.4,1.4-0.5,2.1-0.7 	C17.9,0.5,16.7,1,15.8,0.7z M17.2,4.8c0-0.1,0-0.2,0-0.2c0.2,0,0.3,0.1,0.4,0.1C17.5,4.9,17.4,4.9,17.2,4.8z M17.9,1.8 	c0.3,0,0.2-0.1,0.4,0.1C18.3,2.2,17.7,2.1,17.9,1.8z M18.4,0.2c0.2,0,0.2-0.1,0.4,0c0,0.1-0.1,0.1-0.1,0.1c0.1,0.1,0.3,0,0.2,0.4 	c0.1-0.1,0.4,0,0.6,0c-0.1,0.4-0.7,0.1-1.1,0.1C18.5,0.4,18.4,0.3,18.4,0.2z M21,4.8C20.2,5,18.7,5.1,18,4.7 	C18.9,4.4,20.2,4.5,21,4.8z M20.1,0.8c0-0.3-0.4-0.5-0.1-0.7c0.4,0.1,0.7,0,1,0.1c0.1,0.2-0.4,0.4-0.3,0.7 	C20.6,0.9,20.4,0.9,20.1,0.8z M22.1,9.5c0.1,0.3-0.7,0.5-1.1,0.2c-0.7,0.3-1.7,0.2-2.8,0.1C19,8.9,20.9,9.9,22.1,9.5z M20.5,2 	c0.3-0.1,0.5-0.2,0.9-0.2C21.6,2.1,20.7,2.2,20.5,2z M21.2,0.9c0.1-0.3,0.6-0.3,0.7-0.7c1,0.1,2.2,0,2.9,0.1 	c-0.2,0.3-1.1,0.2-0.9,0.6C23.2,0.6,22.3,0.9,21.2,0.9z M20.5,7.1c0.1-0.2,0.3-0.2,0.4-0.4c0.5,0.1,1,0.1,1.2,0.3 	C21.7,7.4,20.9,7.2,20.5,7.1z M21.6,4.8c0.4-0.1,1-0.4,1.4-0.2C22.7,4.9,22,5,21.6,4.8z M24.3,2.1c-0.5,0.1-1.3,0.1-1.9,0.1 	C22.3,1.8,23.9,1.7,24.3,2.1z M23.2,7.1c0.3-0.2,1-0.4,1.6-0.1C24.5,7.2,23.6,7.5,23.2,7.1z M24.4,9.9c-0.4,0.3-1.1,0-1.4-0.3 	C23.5,9.5,24.1,9.8,24.4,9.9z M25.3,0.7c0-0.1,0-0.1,0-0.2c0.1,0,0.1,0,0.1-0.1c0.1,0,0.1,0,0.2,0c0,0.1,0,0.1,0.1,0.1 	c0,0.1,0,0.1,0,0.2c-0.1,0-0.1,0-0.1,0.1C25.6,0.8,25.5,0.8,25.3,0.7C25.5,0.8,25.4,0.7,25.3,0.7z M25.8,0.5 	c-0.2-0.2,0.3-0.4,0.6-0.3C26.4,0.5,25.9,0.4,25.8,0.5z M24.9,7c0.9-0.5,2.9-0.2,4.2,0C28.7,7.5,27.4,7,27,7.4 	C26.3,7,25.4,7.5,24.9,7z M26.5,9.9c-0.6,0.4-1.6-0.1-2,0C25,9.5,26.1,9.9,26.5,9.9z M27.3,0.4c0-0.3,0.2-0.3,0.5-0.3 	C27.8,0.4,27.6,0.5,27.3,0.4z M28.4,2.8c0.5-0.5,1.6-0.1,2,0C30,3.3,28.8,2.7,28.4,2.8z M28.5,9.9c-0.2,0.2-0.7,0.3-0.8,0.1 	C27.8,9.8,28.3,9.8,28.5,9.9z M36.6,1.2c0.2-0.2,0.6-0.5,0.9-0.3C37.2,1,37,1.2,36.6,1.2z M37.8,0.5c0.1-0.1,0.1-0.1,0-0.2 	c0.1-0.1,0.4-0.1,0.6-0.1C38.4,0.5,38.1,0.5,37.8,0.5z M39.6,0.4c0.5-0.2,1.9-0.2,2.7,0c1.1-0.3,2.8-0.1,4.3,0 	c-0.9,0.7-2.6,1-4.2,0.6c0.1-0.3,0.7-0.1,0.9-0.3C42,0.1,40.7,0.8,39.6,0.4z M41.4,0.9c-0.1,0.2-0.4,0.2-0.7,0.1 	C40.7,0.6,41.3,0.7,41.4,0.9z M52,10.5c-0.1,0.1-0.1,0.1,0,0.2c-0.2,0.1-0.5,0.1-0.8,0C51.1,10.6,51.7,10.5,52,10.5z M64.6,10.1 	c0,0.4-0.3,0.1-0.6,0.1c-1.9,0.4-4.7,0.6-6.7,0.2c-0.8,0.2-1.7,0.4-2.8,0.3c0-0.1,0.1-0.1,0.1-0.2c0.8-0.1,1.9,0,2.5-0.3 	c0.3,0,0.5,0.1,0.8,0.1c1.8,0.1,4.3-0.3,5.9-0.3C64,9.9,64.9,9.9,64.6,10.1z M84.4,4.5c0-0.3-0.2-0.6,0-0.7c0.1,0.1,0.1,0.1,0.3,0.1 	C84.6,4.2,85,4.6,84.4,4.5z M7.2,6.4C7,6.5,6.8,6.5,6.6,6.4c0.1-0.2,0.4,0,0.4-0.2C7.2,6.3,7,6.3,7.2,6.4z M9.3,4.5 	c0.2-0.3,1,0.1,1.4,0C10.5,4.6,9.7,4.7,9.3,4.5z M12.8,9.3c-0.2,0.2-0.6,0-0.9-0.1c0.2-0.1,0.6-0.2,0.7,0c0.1,0,0.1,0,0.1-0.1 	C12.9,9.1,12.6,9.2,12.8,9.3z M19.5,6.1c0-0.2-0.2-0.2-0.1-0.4c0.2-0.1,0.7,0,0.9,0c-0.2,0.3,0.5,0.2,0.4,0.5 	C20.3,5.6,20.2,6.3,19.5,6.1z M25.7,4.7c0.4-0.5,1.2-0.1,1.7-0.1c0.2,0.1-0.6,0-0.7,0.2c-0.3,0-0.2-0.3-0.4-0.3 	C26.1,4.6,26.1,4.9,25.7,4.7z M87.2,7.6c-0.6,0.7-2.4,0-2.9,0.7c-5.2-0.2-9.2,0.1-14.5-0.2c-0.3,0.1,0.3,0.2,0.1,0.5 	c-0.7,0.2-1.9-0.1-2.5,0.1c0.6,0.5,2.5,0,3,0.4c-0.1,0.4-0.8,0.2-1.3,0.2c-2,0.5-4.3,0.1-6.6,0.1c-1.5,0-3.2,0.1-4.8,0.1 	c-6.2,0.1-12.3,0.3-19,0.6C38.4,10.3,38.5,10,38,10c-1.4-0.1-2.9,0.5-4.4,0.4c-1.2-0.1-2.8-0.5-3.9-0.3c0.2-0.4,1.4-0.2,1.8-0.1 	c3.5-0.6,7.5-0.3,10.8-0.3c1-0.4,2.8,0.1,3.7-0.5c0-0.1-0.1-0.1-0.1-0.2c-0.4-0.3-0.9,0.2-1.4,0.1c-0.3,0-0.1-0.3-0.3-0.4 	c-1,0.1-1.9-0.2-3.1,0c0,0.4,0.5,0,0.7,0.3c-0.9,0.1-2.6,0.6-2.7-0.4c0.1,0.1,0.5,0.2,0.2,0.4c0.5-0.1,0.9-0.1,1.4-0.2 	c-0.1-0.4-0.9,0-1.3,0c-0.1-0.2-0.4-0.2-0.5-0.4C35,8.2,30.5,8.7,26.2,7.9c2.2-0.3,5.2,0.1,7.6-0.4c0.5,0.7,1.7,0.2,2.5,0.3 	c-0.2-0.5-1.3-0.4-1.7-0.2c0.1-0.3-0.6-0.1-0.4-0.4c-1.6-0.1-3.3,0.7-4.7-0.3c0.5-0.3,1.3-0.2,2-0.3c-1.8-0.7-3.7-0.1-5.5-0.2 	c-0.4,0-0.9-0.3-1.4-0.3c-0.4,0-0.8,0.2-1.2,0.2c-0.8,0-1.6-0.5-2.5-0.4c0.1-0.4,0.7-0.2,1.2-0.2c1.4-0.4,2.9-0.1,4.3-0.3 	C27.8,5.4,29,4.6,30.3,5c0.2-0.2,0.5-0.4,0.9-0.4c0,0.2,0.2,0.3,0.6,0.3c0.1-0.2-0.2-0.2-0.3-0.3c0.3-0.3,0.6,0,0.9,0.1 	c0.2,0-0.1-0.3,0.3-0.3c0.4,0,0.3,0.2,0.4,0.3c1.5-0.3,3.3,0.3,4.5-0.3c0.4,0,0.6,0.1,0.9,0.2c0.2,0,0.2-0.3,0.4-0.4 	c3.3,0.3,6.7,0,10.1,0c-0.4-0.2-1.5,0-2.2-0.4c-0.5,0-2.2,0.3-2.5,0.2c-0.1,0-0.2-0.3-0.4-0.3c-0.9-0.1-1.5,0.5-2.7,0.3 	c-0.3,0-0.1-0.3-0.3-0.4c-0.9,0-2,0.1-3.4,0c-0.7,0-1.9-0.3-2.1,0c0,0,0.2,0.2,0.3,0.1c-0.3,0.1-0.8,0.1-1,0.2 	c0-0.3,0.3-0.3,0.1-0.5c-0.9,0.1-1.1-0.3-1.7-0.4c0-0.7,1.2,0,1.4-0.5c-0.8-0.3-2.6-0.6-3.5,0c-1.7-0.9-4,0-5.7-0.4 	c0.4-0.5,1.6-0.5,2.3-0.4c1.3-0.5,3,0.2,4.2-0.3c0.4,0.5,1.1,0.1,1.7,0.4c1.6-0.9,4,0.2,5.8-0.5c0.2,0.1,0.2,0.2,0.5,0.2 	c0.8-0.2,2,0,2.7-0.4c0.8,0.2,1.4-0.1,2,0.2c0.5-0.1,1-0.2,1.7-0.1c0.6-0.4,1.8-0.2,2.3-0.7c-0.4-0.3-0.9,0.2-1.5,0 	c-0.1-0.1,0-0.3-0.2-0.4c2.5-0.4,5.4,0,8.2-0.1c0.2,0.1,0.5,0.2,0.3,0.3c0.4,0,0.5-0.3,0.8-0.3c2.6-0.2,5.3,0.1,7.8,0.3 	c1.8,0.1,3.6-0.2,5.1,0.6c1.4,0.1,3.2,0,4.9,0.5c0.8,0.2,1.3,0.6,2.1,0.6c0.5,0,1.1-0.2,1.6-0.1c0.2,0.1,0.1,0.4,0.2,0.5 	c0.8,0.1,1.5-0.3,2.1,0c0,0.2-0.3,0.1-0.4,0.4C79.7,3,80,3.1,80.2,2.9C80.6,4.3,83.3,4,84,5.2c1.5,0.1,3.9,0.5,4,1.5 	c-0.6-0.2-1.5-0.8-2.2-0.4C85.9,6.9,87.3,7.1,87.2,7.6z M35.7,0.1c0.2,0-0.1,0.1,0.1,0.2c-0.2,0.1-0.6,0.2-0.8,0.2 	C34.2,0.5,34.6,0,35.7,0.1z M53.2,10.6c0.3-0.2,0.7-0.2,1.1-0.1c-0.1,0.4-0.9,0.1-1.3,0.2C53.2,10.6,53.4,10.6,53.2,10.6z M27.3,0.8 	c1.6-0.9,5.1-0.7,7.1-0.6c-1.3,0.6-3.4,1.1-5.2,0.6c0.1-0.2,0.5,0,0.7-0.2C28.9,0.2,28.1,1.3,27.3,0.8z"/></svg></div>';
        }
        print( title_html );
        #>
        <?php
    }
}

$widgets_manager->register_widget_type(new OSF_Elementor_Heading());